//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	RAD Verification Regression Analysis
// Objective: 	Examining the change in wealth for respondents who moved into a
//				nursing home to determine whether or not they are including the
//				value of RADs as part of their wealth holdings.
//
// Created: 	12/05/2021
// Modified: 	15/21/2021
//==============================================================================

**************************
*** Importing the Data ***
**************************

clear

cd "XXXX" // set the content directory here

use HILDA_restricted_combined_rescaled	// importing the data

xtset xwaveid wave	// declaring the data to be a panel

merge m:1 year using CPI_2018.dta, nogen

drop if (year == 2000 | year == 2020)

****************************
*** Generating Variables ***
****************************

gen net_worth = pwassei - pwdebti	// generating a net worth variable
gen income = tifefp - tifefn

local money_vars income net_worth

foreach i in `money_vars' {
	replace `i' = `i' * (100/cpi)
}

xtset xwaveid wave	// declaring the data to be a panel

// generating a variable equal to the sum of income received in the preceeding four years
gen sum_income = income + L.income + L2.income + L3.income
replace sum_income = income + L.income + L2.income if sum_income == .
replace sum_income = income + L.income if sum_income == .
replace sum_income = income if sum_income == .

// generating a variable equal to the length of time an individual has spent in a nursing home
gen duration_nursing_home = 0
replace duration_nursing_home = 1 if dodtyp == 1
replace duration_nursing_home = 2 if dodtyp == 1 & L.dodtyp == 1
replace duration_nursing_home = 3 if dodtyp == 1 & L2.dodtyp == 1
replace duration_nursing_home = 4 if dodtyp == 1 & L3.dodtyp == 1
replace duration_nursing_home = 5 if dodtyp == 1 & L4.dodtyp == 1
replace duration_nursing_home = 6 if dodtyp == 1 & L5.dodtyp == 1
replace duration_nursing_home = 7 if dodtyp == 1 & L6.dodtyp == 1
replace duration_nursing_home = 8 if dodtyp == 1 & L7.dodtyp == 1
replace duration_nursing_home = 9 if dodtyp == 1 & L8.dodtyp == 1
replace duration_nursing_home = 10 if dodtyp == 1 & L9.dodtyp == 1
replace duration_nursing_home = 11 if dodtyp == 1 & L10.dodtyp == 1
replace duration_nursing_home = 12 if dodtyp == 1 & L11.dodtyp == 1
replace duration_nursing_home = duration_nursing_home - 0.5 if duration_nursing_home > 0

// dropping observations for which wealth data is unavailable
drop if (wave != 2 & wave != 6 & wave != 10 & wave != 14 & wave != 18)

gen module = 1
replace module = 2 if wave == 6
replace module = 3 if wave == 10
replace module = 4 if wave == 14
replace module = 5 if wave == 18

xtset xwaveid module	// declaring the data to be a panel

// generating a gender variable
codebook sex
gen female = 0
replace female = 1 if sex == 2

// generating age and age-squared variables
codebook hgage
gen age = hgage
gen age2 = hgage ^ 2

// generating a health variable
codebook gh1
gen health = gh1
replace health = . if health <= 0

// generating a variable indicating whether someone is part of a couple
destring hhpxid, replace
gen couple = 0
replace couple = 1 if hhpxid != .

// generating a variable indicating whether someone has moved into a nursing home since the previous wealth module
codebook dodtyp if dodtyp == 1
gen change_nursing_home = 0
replace change_nursing_home = 1 if dodtyp == 1 & L.dodtyp != 1

// generating a previous net worth variable
gen prev_net_worth = L.net_worth

// generating time dummy variables
gen t2002 = 0
replace t2002 = 1 if module == 1
gen t2006 = 0
replace t2006 = 1 if module == 2
gen t2010 = 0
replace t2010 = 1 if module == 3
gen t2014 = 0
replace t2014 = 1 if module == 4
gen t2018 = 0
replace t2018 = 1 if module == 5

***************************
*** Regression Analysis ***
***************************

// generating a change in net worth variable
gen change_net_worth = net_worth - prev_net_worth

// generating property equity variables
gen prop_equity = pwtpvi - pwtpdi
gen change_prop_equity = prop_equity - L.prop_equity
gen prev_prop_equity = L.prop_equity

*****************
*** Net Worth ***
*****************

reg net_worth prev_net_worth sum_income change_nursing_home duration_nursing_home age age2 female couple health t2010 t2014 t2018, vce(cluster xwaveid)	// OLS regression
// reg change_net_worth prev_net_worth sum_income change_nursing_home duration_nursing_home age age2 female couple health	// OLS regression
xtreg net_worth prev_net_worth sum_income change_nursing_home duration_nursing_home i.module age age2 female couple health, fe	// fixed effects regression

***********************
*** Property Equity ***
***********************

reg prop_equity prev_prop_equity sum_income change_nursing_home duration_nursing_home age age2 female couple health	// OLS regression
// reg change_net_worth prev_net_worth sum_income change_nursing_home duration_nursing_home age age2 female couple health	// OLS regression
xtreg prop_equity prev_prop_equity sum_income change_nursing_home duration_nursing_home i.module age age2 female couple health, fe	// fixed effects regression

*****************************
*** Home Ownership Tables ***
*****************************

tab hstenr if dodtyp == 1	// tabulating answers to questions about home ownership
tab hsfrea if dodtyp == 1	// tabulating answers to questions about how housing is provided if not an owner or renting